import React, { useState, useEffect } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { Sidebar } from './components/sidebar';
import { TopBar } from './components/top-bar';
import { MainContent } from './components/main-content';
import { Toaster } from './components/ui/toaster';
import { useToast } from './components/ui/use-toast';
import { StorageManager, STORAGE_KEYS, DataMigration } from './utils/storage';
import { ErrorBoundary } from './components/error-boundary';
import './globals.css';

export interface Student {
  id: string;
  name: string;
  avatar?: string;
  group?: string;
  weight?: number;
}

export interface PickResult {
  students: Student[];
  timestamp: Date;
  mode: 'single' | 'multiple' | 'group';
  context?: string;
}

export interface Settings {
  soundEnabled: boolean;
  soundType: 'bell' | 'applause' | 'music';
  volume: number;
  animationEnabled: boolean;
  animationSpeed: 'slow' | 'normal' | 'fast';
  animationType: 'fade' | 'rotate' | 'scale';
  theme: 'light' | 'dark' | 'auto';
  language: 'zh-CN' | 'en-US';
}

function App() {
  const defaultSettings: Settings = {
    soundEnabled: true,
    soundType: 'bell',
    volume: 0.7,
    animationEnabled: true,
    animationSpeed: 'normal',
    animationType: 'fade',
    theme: 'light',
    language: 'zh-CN'
  };

  // 状态管理
  const [activeSection, setActiveSection] = useState('dashboard');
  const [sidebarCollapsed, setSidebarCollapsed] = useState(false);
  const [students, setStudents] = useState<Student[]>(() => 
    StorageManager.get(STORAGE_KEYS.STUDENTS, [])
  );
  const [currentResult, setCurrentResult] = useState<PickResult | null>(null);
  const [history, setHistory] = useState<PickResult[]>(() => 
    StorageManager.get(STORAGE_KEYS.HISTORY, [])
  );
  const [settings, setSettings] = useState<Settings>(() => 
    StorageManager.get(STORAGE_KEYS.SETTINGS, defaultSettings)
  );
  const [isSettingsOpen, setIsSettingsOpen] = useState(false);
  const { toast } = useToast();

  // 数据初始化
  useEffect(() => {
    DataMigration.checkAndMigrate();
    
    const savedStudents = StorageManager.get(STORAGE_KEYS.STUDENTS, []);
    const savedHistory = StorageManager.get(STORAGE_KEYS.HISTORY, []);
    
    if (savedStudents.length > 0 || savedHistory.length > 0) {
      toast({
        title: "数据已恢复",
        description: `已加载 ${savedStudents.length} 个学生，${savedHistory.length} 条历史记录`,
      });
    }
  }, [toast]);

  // 自动保存
  useEffect(() => {
    StorageManager.set(STORAGE_KEYS.STUDENTS, students);
  }, [students]);

  useEffect(() => {
    StorageManager.set(STORAGE_KEYS.HISTORY, history);
  }, [history]);

  useEffect(() => {
    StorageManager.set(STORAGE_KEYS.SETTINGS, settings);
  }, [settings]);

  // 键盘快捷键
  useEffect(() => {
    const handleKeyDown = (e: KeyboardEvent) => {
      if (e.target instanceof HTMLInputElement || e.target instanceof HTMLTextAreaElement) {
        return;
      }

      switch (e.key) {
        case 'Escape':
          if (isSettingsOpen) setIsSettingsOpen(false);
          break;
        case 's':
        case 'S':
          if (e.ctrlKey || e.metaKey) {
            e.preventDefault();
            setIsSettingsOpen(true);
          }
          break;
        case 'h':
        case 'H':
          if (e.ctrlKey || e.metaKey) {
            e.preventDefault();
            window.open('/help.html', '_blank');
          }
          break;
      }
    };

    document.addEventListener('keydown', handleKeyDown);
    return () => document.removeEventListener('keydown', handleKeyDown);
  }, []);

  const handleStudentsImport = (importedStudents: Student[]) => {
    setStudents(importedStudents);
  };

  const handlePickResult = (result: PickResult) => {
    setCurrentResult(result);
    setHistory(prev => [result, ...prev].slice(0, 100));
  };

  const handleRemoveStudent = (studentId: string) => {
    setStudents(prev => prev.filter(s => s.id !== studentId));
  };

  const handleClearStudents = () => {
    setStudents([]);
    setCurrentResult(null);
  };

  return (
    <ErrorBoundary>
      <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100">
          {/* 侧边栏 */}
          <ErrorBoundary>
            <Sidebar
              activeSection={activeSection}
              onSectionChange={setActiveSection}
              collapsed={sidebarCollapsed}
              onToggleCollapse={() => setSidebarCollapsed(!sidebarCollapsed)}
            />
          </ErrorBoundary>

          {/* 主内容区域 */}
          <div className={`transition-all duration-300 ${sidebarCollapsed ? 'ml-16' : 'ml-72'}`}>
            {/* 顶部导航栏 */}
            <ErrorBoundary>
            <TopBar
              activeSection={activeSection}
              studentsCount={students.length}
              sidebarCollapsed={sidebarCollapsed}
              onToggleSidebar={() => setSidebarCollapsed(!sidebarCollapsed)}
            />
            </ErrorBoundary>

            {/* 主要内容 */}
            <main className="p-6">
              <AnimatePresence mode="wait">
                <motion.div
                  key={activeSection}
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  exit={{ opacity: 0, y: -20 }}
                  transition={{ duration: 0.3, ease: "easeInOut" }}
                >
                <MainContent
                  activeSection={activeSection}
                  students={students}
                  currentResult={currentResult}
                  history={history}
                  settings={settings}
                  onStudentsImport={handleStudentsImport}
                  onPickResult={handlePickResult}
                  onRemoveStudent={handleRemoveStudent}
                  onClearStudents={handleClearStudents}
                  onSettingsChange={setSettings}
                  onSectionChange={setActiveSection}
                />
                </motion.div>
              </AnimatePresence>
            </main>
          </div>

          <Toaster />
        </div>
      </ErrorBoundary>
    );
}

export default App;