import React, { useState, useEffect } from 'react';
import { Button } from './ui/button';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Textarea } from './ui/textarea';
import { Badge } from './ui/badge';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from './ui/dialog';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { 
  Calendar, 
  Clock, 
  Plus, 
  Edit, 
  Trash2, 
  BookOpen, 
  MapPin, 
  User,
  Download,
  Upload,
  RefreshCw
} from 'lucide-react';
import { cn } from '../lib/utils';
import { useToast } from '../hooks/use-toast';

interface Course {
  id: string;
  name: string;
  teacher: string;
  location: string;
  color: string;
  description?: string;
}

interface ScheduleItem {
  id: string;
  courseId: string;
  dayOfWeek: number; // 0-6 (周日-周六)
  startTime: string; // HH:MM
  endTime: string; // HH:MM
  weeks: number[]; // 周次数组
}

interface ScheduleProps {
  className?: string;
}

const DAYS = ['周日', '周一', '周二', '周三', '周四', '周五', '周六'];
const TIME_SLOTS = [
  '08:00', '08:30', '09:00', '09:30', '10:00', '10:30', '11:00', '11:30',
  '12:00', '12:30', '13:00', '13:30', '14:00', '14:30', '15:00', '15:30',
  '16:00', '16:30', '17:00', '17:30', '18:00', '18:30', '19:00', '19:30',
  '20:00', '20:30', '21:00', '21:30'
];

const COLORS = [
  '#3B82F6', '#EF4444', '#10B981', '#F59E0B', '#8B5CF6',
  '#EC4899', '#06B6D4', '#84CC16', '#F97316', '#6366F1'
];

export default function Schedule({ className }: ScheduleProps) {
  const [courses, setCourses] = useState<Course[]>([]);
  const [scheduleItems, setScheduleItems] = useState<ScheduleItem[]>([]);
  const [currentWeek, setCurrentWeek] = useState(1);
  const [totalWeeks, setTotalWeeks] = useState(20);
  const [isAddingCourse, setIsAddingCourse] = useState(false);
  const [isAddingSchedule, setIsAddingSchedule] = useState(false);
  const [editingCourse, setEditingCourse] = useState<Course | null>(null);
  const { toast } = useToast();

  // 新课程表单
  const [newCourse, setNewCourse] = useState({
    name: '',
    teacher: '',
    location: '',
    color: COLORS[0],
    description: ''
  });

  // 新课程安排表单
  const [newSchedule, setNewSchedule] = useState({
    courseId: '',
    dayOfWeek: 1,
    startTime: '09:00',
    endTime: '10:30',
    weeks: [] as number[]
  });

  // 从本地存储加载数据
  useEffect(() => {
    const savedCourses = localStorage.getItem('schedule_courses');
    const savedScheduleItems = localStorage.getItem('schedule_items');
    const savedCurrentWeek = localStorage.getItem('schedule_current_week');
    const savedTotalWeeks = localStorage.getItem('schedule_total_weeks');

    if (savedCourses) setCourses(JSON.parse(savedCourses));
    if (savedScheduleItems) setScheduleItems(JSON.parse(savedScheduleItems));
    if (savedCurrentWeek) setCurrentWeek(Number(savedCurrentWeek));
    if (savedTotalWeeks) setTotalWeeks(Number(savedTotalWeeks));
  }, []);

  // 保存数据到本地存储
  useEffect(() => {
    localStorage.setItem('schedule_courses', JSON.stringify(courses));
  }, [courses]);

  useEffect(() => {
    localStorage.setItem('schedule_items', JSON.stringify(scheduleItems));
  }, [scheduleItems]);

  useEffect(() => {
    localStorage.setItem('schedule_current_week', currentWeek.toString());
  }, [currentWeek]);

  useEffect(() => {
    localStorage.setItem('schedule_total_weeks', totalWeeks.toString());
  }, [totalWeeks]);

  // 添加课程
  const addCourse = () => {
    if (!newCourse.name.trim()) {
      toast({
        title: "错误",
        description: "请输入课程名称",
        variant: "destructive"
      });
      return;
    }

    const course: Course = {
      id: Date.now().toString(),
      ...newCourse,
      color: newCourse.color || COLORS[courses.length % COLORS.length]
    };

    setCourses(prev => [...prev, course]);
    setNewCourse({
      name: '',
      teacher: '',
      location: '',
      color: COLORS[0],
      description: ''
    });
    setIsAddingCourse(false);
    
    toast({
      title: "成功",
      description: "课程添加成功"
    });
  };

  // 删除课程
  const deleteCourse = (courseId: string) => {
    setCourses(prev => prev.filter(course => course.id !== courseId));
    setScheduleItems(prev => prev.filter(item => item.courseId !== courseId));
    
    toast({
      title: "成功",
      description: "课程删除成功"
    });
  };

  // 添加课程安排
  const addScheduleItem = () => {
    if (!newSchedule.courseId) {
      toast({
        title: "错误",
        description: "请选择课程",
        variant: "destructive"
      });
      return;
    }

    if (newSchedule.weeks.length === 0) {
      toast({
        title: "错误",
        description: "请选择上课周次",
        variant: "destructive"
      });
      return;
    }

    const scheduleItem: ScheduleItem = {
      id: Date.now().toString(),
      ...newSchedule
    };

    setScheduleItems(prev => [...prev, scheduleItem]);
    setNewSchedule({
      courseId: '',
      dayOfWeek: 1,
      startTime: '09:00',
      endTime: '10:30',
      weeks: []
    });
    setIsAddingSchedule(false);
    
    toast({
      title: "成功",
      description: "课程安排添加成功"
    });
  };

  // 删除课程安排
  const deleteScheduleItem = (itemId: string) => {
    setScheduleItems(prev => prev.filter(item => item.id !== itemId));
    
    toast({
      title: "成功",
      description: "课程安排删除成功"
    });
  };

  // 获取指定时间段的课程
  const getCoursesForTimeSlot = (dayOfWeek: number, timeSlot: string) => {
    return scheduleItems.filter(item => {
      if (item.dayOfWeek !== dayOfWeek) return false;
      if (!item.weeks.includes(currentWeek)) return false;
      
      const itemStart = item.startTime;
      const itemEnd = item.endTime;
      const slotTime = timeSlot;
      
      return slotTime >= itemStart && slotTime < itemEnd;
    }).map(item => {
      const course = courses.find(c => c.id === item.courseId);
      return { ...item, course };
    });
  };

  // 生成周次选择器
  const generateWeekSelector = () => {
    const weeks = [];
    for (let i = 1; i <= totalWeeks; i++) {
      weeks.push(i);
    }
    return weeks;
  };

  // 导出课程表
  const exportSchedule = () => {
    const data = {
      courses,
      scheduleItems,
      currentWeek,
      totalWeeks,
      exportTime: new Date().toISOString()
    };
    
    const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `课程表_${new Date().toLocaleDateString().replace(/\//g, '-')}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
    
    toast({
      title: "成功",
      description: "课程表导出成功"
    });
  };

  // 导入课程表
  const importSchedule = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    const reader = new FileReader();
    reader.onload = (event) => {
      try {
        const data = JSON.parse(event.target?.result as string);
        if (data.courses) setCourses(data.courses);
        if (data.scheduleItems) setScheduleItems(data.scheduleItems);
        if (data.currentWeek) setCurrentWeek(data.currentWeek);
        if (data.totalWeeks) setTotalWeeks(data.totalWeeks);
        
        toast({
          title: "成功",
          description: "课程表导入成功"
        });
      } catch (error) {
        toast({
          title: "错误",
          description: "文件格式错误",
          variant: "destructive"
        });
      }
    };
    reader.readAsText(file);
  };

  return (
    <div className={cn("space-y-6", className)}>
      {/* 头部控制区 */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Calendar className="h-5 w-5" />
            课程表管理
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-wrap items-center gap-4">
            {/* 周次控制 */}
            <div className="flex items-center gap-2">
              <Label>当前周次:</Label>
              <Select value={currentWeek.toString()} onValueChange={(value) => setCurrentWeek(Number(value))}>
                <SelectTrigger className="w-20">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {generateWeekSelector().map(week => (
                    <SelectItem key={week} value={week.toString()}>
                      第{week}周
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div className="flex items-center gap-2">
              <Label>总周数:</Label>
              <Input
                type="number"
                min="1"
                max="30"
                value={totalWeeks}
                onChange={(e) => setTotalWeeks(Number(e.target.value))}
                className="w-20"
              />
            </div>

            {/* 操作按钮 */}
            <div className="flex gap-2 ml-auto">
              <Dialog open={isAddingCourse} onOpenChange={setIsAddingCourse}>
                <DialogTrigger asChild>
                  <Button variant="outline" size="sm">
                    <Plus className="h-4 w-4 mr-1" />
                    添加课程
                  </Button>
                </DialogTrigger>
                <DialogContent>
                  <DialogHeader>
                    <DialogTitle>添加新课程</DialogTitle>
                  </DialogHeader>
                  <div className="space-y-4">
                    <div>
                      <Label>课程名称</Label>
                      <Input
                        value={newCourse.name}
                        onChange={(e) => setNewCourse(prev => ({ ...prev, name: e.target.value }))}
                        placeholder="请输入课程名称"
                      />
                    </div>
                    <div>
                      <Label>授课教师</Label>
                      <Input
                        value={newCourse.teacher}
                        onChange={(e) => setNewCourse(prev => ({ ...prev, teacher: e.target.value }))}
                        placeholder="请输入教师姓名"
                      />
                    </div>
                    <div>
                      <Label>上课地点</Label>
                      <Input
                        value={newCourse.location}
                        onChange={(e) => setNewCourse(prev => ({ ...prev, location: e.target.value }))}
                        placeholder="请输入上课地点"
                      />
                    </div>
                    <div>
                      <Label>课程颜色</Label>
                      <div className="flex gap-2 mt-2">
                        {COLORS.map(color => (
                          <button
                            key={color}
                            className={cn(
                              "w-8 h-8 rounded border-2",
                              newCourse.color === color ? "border-gray-400" : "border-gray-200"
                            )}
                            style={{ backgroundColor: color }}
                            onClick={() => setNewCourse(prev => ({ ...prev, color }))}
                          />
                        ))}
                      </div>
                    </div>
                    <div>
                      <Label>课程描述</Label>
                      <Textarea
                        value={newCourse.description}
                        onChange={(e) => setNewCourse(prev => ({ ...prev, description: e.target.value }))}
                        placeholder="请输入课程描述（可选）"
                      />
                    </div>
                    <div className="flex justify-end gap-2">
                      <Button variant="outline" onClick={() => setIsAddingCourse(false)}>
                        取消
                      </Button>
                      <Button onClick={addCourse}>
                        添加课程
                      </Button>
                    </div>
                  </div>
                </DialogContent>
              </Dialog>

              <Dialog open={isAddingSchedule} onOpenChange={setIsAddingSchedule}>
                <DialogTrigger asChild>
                  <Button variant="outline" size="sm">
                    <Clock className="h-4 w-4 mr-1" />
                    添加课程安排
                  </Button>
                </DialogTrigger>
                <DialogContent>
                  <DialogHeader>
                    <DialogTitle>添加课程安排</DialogTitle>
                  </DialogHeader>
                  <div className="space-y-4">
                    <div>
                      <Label>选择课程</Label>
                      <Select value={newSchedule.courseId} onValueChange={(value) => setNewSchedule(prev => ({ ...prev, courseId: value }))}>
                        <SelectTrigger>
                          <SelectValue placeholder="请选择课程" />
                        </SelectTrigger>
                        <SelectContent>
                          {courses.map(course => (
                            <SelectItem key={course.id} value={course.id}>
                              {course.name}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                    <div>
                      <Label>星期</Label>
                      <Select value={newSchedule.dayOfWeek.toString()} onValueChange={(value) => setNewSchedule(prev => ({ ...prev, dayOfWeek: Number(value) }))}>
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          {DAYS.map((day, index) => (
                            <SelectItem key={index} value={index.toString()}>
                              {day}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <Label>开始时间</Label>
                        <Select value={newSchedule.startTime} onValueChange={(value) => setNewSchedule(prev => ({ ...prev, startTime: value }))}>
                          <SelectTrigger>
                            <SelectValue />
                          </SelectTrigger>
                          <SelectContent>
                            {TIME_SLOTS.map(time => (
                              <SelectItem key={time} value={time}>
                                {time}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                      </div>
                      <div>
                        <Label>结束时间</Label>
                        <Select value={newSchedule.endTime} onValueChange={(value) => setNewSchedule(prev => ({ ...prev, endTime: value }))}>
                          <SelectTrigger>
                            <SelectValue />
                          </SelectTrigger>
                          <SelectContent>
                            {TIME_SLOTS.map(time => (
                              <SelectItem key={time} value={time}>
                                {time}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                      </div>
                    </div>
                    <div>
                      <Label>上课周次</Label>
                      <div className="grid grid-cols-5 gap-2 mt-2 max-h-32 overflow-y-auto">
                        {generateWeekSelector().map(week => (
                          <label key={week} className="flex items-center space-x-2">
                            <input
                              type="checkbox"
                              checked={newSchedule.weeks.includes(week)}
                              onChange={(e) => {
                                if (e.target.checked) {
                                  setNewSchedule(prev => ({ ...prev, weeks: [...prev.weeks, week] }));
                                } else {
                                  setNewSchedule(prev => ({ ...prev, weeks: prev.weeks.filter(w => w !== week) }));
                                }
                              }}
                            />
                            <span className="text-sm">第{week}周</span>
                          </label>
                        ))}
                      </div>
                    </div>
                    <div className="flex justify-end gap-2">
                      <Button variant="outline" onClick={() => setIsAddingSchedule(false)}>
                        取消
                      </Button>
                      <Button onClick={addScheduleItem}>
                        添加安排
                      </Button>
                    </div>
                  </div>
                </DialogContent>
              </Dialog>

              <Button variant="outline" size="sm" onClick={exportSchedule}>
                <Download className="h-4 w-4 mr-1" />
                导出
              </Button>

              <label className="cursor-pointer">
                <Button variant="outline" size="sm" asChild>
                  <span>
                    <Upload className="h-4 w-4 mr-1" />
                    导入
                  </span>
                </Button>
                <input
                  type="file"
                  accept=".json"
                  onChange={importSchedule}
                  className="hidden"
                />
              </label>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* 课程列表 */}
      <Card>
        <CardHeader>
          <CardTitle>课程列表</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {courses.map(course => (
              <div key={course.id} className="border rounded-lg p-4 space-y-2">
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-2">
                    <div
                      className="w-4 h-4 rounded"
                      style={{ backgroundColor: course.color }}
                    />
                    <h3 className="font-medium">{course.name}</h3>
                  </div>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => deleteCourse(course.id)}
                    className="h-8 w-8 p-0 text-red-500 hover:text-red-700"
                  >
                    <Trash2 className="h-4 w-4" />
                  </Button>
                </div>
                <div className="space-y-1 text-sm text-gray-600">
                  <div className="flex items-center gap-1">
                    <User className="h-3 w-3" />
                    <span>{course.teacher || '未设置'}</span>
                  </div>
                  <div className="flex items-center gap-1">
                    <MapPin className="h-3 w-3" />
                    <span>{course.location || '未设置'}</span>
                  </div>
                  {course.description && (
                    <div className="flex items-start gap-1">
                      <BookOpen className="h-3 w-3 mt-0.5" />
                      <span className="text-xs">{course.description}</span>
                    </div>
                  )}
                </div>
              </div>
            ))}
            {courses.length === 0 && (
              <div className="col-span-full text-center py-8 text-gray-500">
                暂无课程，请先添加课程
              </div>
            )}
          </div>
        </CardContent>
      </Card>

      {/* 课程表 */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center justify-between">
            <span>第{currentWeek}周课程表</span>
            <Badge variant="outline">
              共{totalWeeks}周
            </Badge>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="overflow-x-auto">
            <table className="w-full border-collapse border border-gray-300">
              <thead>
                <tr>
                  <th className="border border-gray-300 p-2 bg-gray-50 w-20">时间</th>
                  {DAYS.slice(1, 6).map((day, index) => (
                    <th key={day} className="border border-gray-300 p-2 bg-gray-50">
                      {day}
                    </th>
                  ))}
                </tr>
              </thead>
              <tbody>
                {TIME_SLOTS.filter((_, index) => index % 2 === 0).map((timeSlot, timeIndex) => (
                  <tr key={timeSlot}>
                    <td className="border border-gray-300 p-2 text-center text-sm bg-gray-50">
                      {timeSlot}
                      <br />
                      <span className="text-xs text-gray-500">
                        {TIME_SLOTS[timeIndex * 2 + 1]}
                      </span>
                    </td>
                    {[1, 2, 3, 4, 5].map(dayOfWeek => {
                      const coursesInSlot = getCoursesForTimeSlot(dayOfWeek, timeSlot);
                      return (
                        <td key={dayOfWeek} className="border border-gray-300 p-1 h-20 align-top">
                          {coursesInSlot.map(item => (
                            <div
                              key={item.id}
                              className="text-xs p-1 rounded mb-1 text-white relative group"
                              style={{ backgroundColor: item.course?.color || '#gray' }}
                            >
                              <div className="font-medium truncate">
                                {item.course?.name}
                              </div>
                              <div className="text-xs opacity-90 truncate">
                                {item.course?.location}
                              </div>
                              <div className="text-xs opacity-75">
                                {item.startTime}-{item.endTime}
                              </div>
                              <Button
                                variant="ghost"
                                size="sm"
                                onClick={() => deleteScheduleItem(item.id)}
                                className="absolute top-0 right-0 h-4 w-4 p-0 opacity-0 group-hover:opacity-100 text-white hover:bg-red-500"
                              >
                                <Trash2 className="h-3 w-3" />
                              </Button>
                            </div>
                          ))}
                        </td>
                      );
                    })}
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
          
          {scheduleItems.length === 0 && (
            <div className="text-center py-8 text-gray-500">
              暂无课程安排，请先添加课程安排
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}